/**
 * @file   zrss.cpp
 * @author NOKUBI Takatsugu
 * @date   Fri Dec 19 14:33:14 2003
 * 
 * @brief  RSS parser class
 * 
 * $Id: zrss.cpp 121 2004-02-17 08:17:49Z knok $
 */

#include "zrss.h"
#include <qregexp.h>

ZRss::ZRss(QString &cont, QString enc)
{
  QRegExp epat("encoding=\"[^\"]*\"", FALSE, FALSE);
  QCString st = cont.replace(epat, "encoding=\"UTF-8\"").utf8();
  QBuffer b(st);
  b.open(IO_ReadOnly);
  QTextStream t(&b);
  QXmlInputSource s(t);
  zh = new ZRssHandler();
  QXmlSimpleReader r;
  ZRssError e;
  r.setFeature("http://xml.org/sax/features/namespaces", true);
  r.setFeature("http://xml.org/sax/features/namespace-prefixes", true);
  r.setFeature("http://trolltech.com/xml/features/report-whitespace-only-CharData", false);
  r.setContentHandler(zh);
  r.setErrorHandler(&e);
  r.parse(s);
  if (enc.isNull()) {
      encoding = "utf-8";
  } else {
      encoding = enc;
  }
}

ZRss::~ZRss()
{
  delete zh;
}

const QString &ZRss::getName()
{
  return zh->getName();
}

const QString &ZRss::getLink()
{
  return zh->getLink();
}

const QString &ZRss::getDesc()
{
  return zh->getDesc();
}

const QString &ZRss::getEncoding()
{
  return encoding;
}

QVector<ZRssItem> & ZRss::getItems()
{
  return zh->items;
}

bool ZRssHandler::startDocument()
{
  mode = None;
  return true;
}

ZRssHandler::ZRssHandler()
  : QXmlDefaultHandler()
{
  items.resize(0);
  citem = NULL;
  mode = None;
}

ZRssHandler::~ZRssHandler()
{
  for (int i = 0; i < items.size(); i ++)
    delete items[i];
  items.resize(0);
}

bool ZRssHandler::startElement(const QString &, const QString &, 
			       const QString &name, const QXmlAttributes &)
{
  printf("startElement: %s\n", name.latin1());
  if (name == "channel") {
    mode = Channel;
  } else if (name == "item") {
    mode = Item;
    citem = new ZRssItem();
    citem->seen = false;
  } else if (name == "title") {
    if (mode == Channel)
	mode = ChannelTitle;
    else
	mode = ItemTitle;
  } else if (name == "link") {
    if (mode == Channel)
	mode = ChannelLink;
    else
	mode = ItemLink;
  }
  return true;
}

bool ZRssHandler::endElement(const QString &, const QString &, 
			     const QString &name)
{
  printf("endElement: %s\n", name.latin1());
  if (name == "channel") {
    mode = None;
  } else if (name == "item") {
    mode = None;
    items.resize(items.size() + 1);
    items.insert(items.size() - 1, citem);
  } else if (name == "title") {
    if (mode == ChannelTitle)
	mode = Channel;
    else
	mode = Item;
  } else if (name == "link") {
    if (mode == ChannelLink)
	mode = Channel;
    else
	mode = Item;
  }
  return true;
}

bool ZRssHandler::characters(const QString &ch)
{
  printf("characters: %s\n", ch.ascii());
  if (mode & Channel) {
    if ((mode & Title)) {
      chName = ch;
    } else if (mode & Link) {
      chLink = ch;
    }
  } else if (mode & Item) {
    if (!citem)
      return true;
    if ((mode & Title)) {
      citem->title = QString(ch);
    } else if (mode & Link) {
      citem->link = ch;
    }
  }
  return true;
}

const QString &ZRssHandler::getName()
{
  return chName;
}

const QString &ZRssHandler::getLink()
{
  return chLink;
}

const QString &ZRssHandler::getDesc()
{
  return chDesc;
}

bool ZRssError::warning(const QXmlParseException &)
{
  err = "warning";
  printf("warning\n");
  return true;
}

bool ZRssError::error(const QXmlParseException &)
{
  err = "error";
  printf("error\n");
  return true;
}

bool ZRssError::fatalError(const QXmlParseException &e)
{
  err = "fatal error: " + e.message();
  printf("fatal error: %s, line %d, col %d\n", e.message().ascii(), e.lineNumber(),
	 e.columnNumber());
  return true;
}

QString ZRssError::errorString()
{
  return err;
}

